%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(240,60,0.3,3,1)
function SEMDOT2D(nelx,nely,vol,rmin,ft)
%% INITIALIZATION
vx=repmat(vol,nely,nelx); vxPhys=vx;
Emin=0.001; maxloop=500; nG=10; ngrid=nG-1; tolx=0.001;
E0=1; nu=0.3; rho =1e-9;
D = 1.0/(1-0.3^2)*[1 0.3 0; 0.3 1 0; 0 0 0.5*(1-0.3)];
[KE,ME] = lk(D,rho);
%% INITIALIZE MMA OPTIMIZER
nele=nely*nelx; m=1; n = nely*nelx; nelm=nely*nelx;
vxmin=1e-3*ones(nelm,1); vxmax=ones(nelm,1);
vxold1=vx(:); vxold2=vxold1;
a0 = 1; ai = 0; ci = 1000; di = 0; low=ones(nelm,1); upp=1;
% PREPARE FINITE ELEMENT ANALYSIS
nodenrs = reshape(1:(1+nelx)*(1+nely),1+nely,1+ nelx);
edofVec = reshape(2*nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat = repmat(edofVec,1,8)+repmat([0 1 2*nely+[2 3 0 1] -2 -1],nelx*nely,1);
iK = reshape(kron(edofMat,ones(8,1))',64*nelx*nely,1);
jK = reshape(kron(edofMat,ones(1,8))',64*nelx*nely,1);
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% INITIALIZE HEAVISIDE SMOOTH FUNCTION
beta=0.5; ER=0.5;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely);
[fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% DEFINE LOADS AND SUPPORTS (CANTILEVER BEAM)
switch(ft)
    case 1 % Pin-pin beam case
        fixeddofs = union(2*(nely+1)-nely-1:2*(nely+1)-nely+1,2*(nely+1)*(nelx+1)-nely-1:2*(nely+1)*(nelx+1)-nely+1);
    case 2 % Fix-pin beam case
        fixeddofs = ([1:2*(nely+1),2*(nely+1)*(nelx+1)-nely]);
    case 3 % Clamped beam case
        fixeddofs = union(1:2*(nely+1),2*(nely+1)*nelx+1:2*(nely+1)*(nelx+1));
end
U = zeros(2*(nely+1)*(nelx+1),1);
alldofs = (1:2*(nely+1)*(nelx+1));
freedofs = setdiff(alldofs,fixeddofs);
change=1;
loop=0;
%% START ITERATION
while (change > tolx || tol>0.001) && loop < maxloop
    loop=loop+1;
    %% FE-ANALYSIS
    sK = KE(:)*vxPhys(:)'+KE(:)*Emin*(1-vxPhys(:))';
    sM = ME(:)*vxPhys(:)'+ME(:)*Emin*(1-vxPhys(:))';
    K = sparse(iK,jK,sK); K = (K+K')/2;
    M = sparse(iK,jK,sM); M = (M+M')/2;
    node_dof=(nelx+1)*(nely+1)+1;
    M(node_dof,node_dof)=M(node_dof,node_dof)+0.0001; %% Lump mass
    [U(freedofs,:),f1]=eigs(K(freedofs,freedofs),M(freedofs,freedofs),1,'sm');
    K1 = reshape(sum((U(edofMat)*KE).*U(edofMat),2),nely,nelx);
    M1 = reshape(sum((U(edofMat)*ME).*U(edofMat),2),nely,nelx);
    dK=(vxPhys+(1-vxPhys)*Emin).*K1;
    dM=(vxPhys+(1-vxPhys)*Emin).*M1;
    c(loop) = -f1; dc = -(dK-f1.*dM);
    dv=ones(nely,nelx);
    %% FILTERING/MODIFICATION OF SENSITIVITIES
    dc(:)=H*(dc(:)./Hs);
    dv(:)=H*(dv(:)./Hs);
    %%  UPDATE DESIGN VARIABLES AND FILTERED ELEMENTAL VOLUME FRACTIONS
    vxval=vx(:);
    f0val=c; df0dx=dc(:);
    fval=sum(vxPhys(:))/(vol*nely*nelx)-1;
    dfdx=dv(:)'/(vol*nely*nelx);
    [vxmma,~,~,~,~,~,~,~,~,low,upp] = ...
        mmasub(m,n,loop,vxval,vxmin,vxmax,vxold1,vxold2,f0val,df0dx,fval,dfdx,low,upp,a0,ai,ci,di);
    vxnew=reshape(vxmma,nely,nelx); vxold2=vxold1(:); vxold1=vx(:);
    vxPhys(:)=(H*vxnew(:))./Hs;
    xn=reshape((Hn*vxPhys(:)./Hns),nely+1,nelx+1);
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH/STEP FUNCTION
    xg=interp2(nodex,nodey,xn,fnx,fny,'linear');
    l1=0; l2=1;
    while (l2-l1)>1e-5
        ls=(l1+l2)/2;
        xgnew=max(0.001,(tanh(beta*ls)+tanh(beta*(xg-ls)))/(tanh(beta*ls)+tanh(beta*(1-ls))));
        if sum(sum(xgnew))/((ngrid*nelx+1)*(ngrid*nely+1))-sum(vxPhys(:))/(nelx*nely)>0, l1=ls; else, l2=ls; end
    end
    %% ASSEMBLE GRID POINTS TO ELEMENTS
    vxPhys(:)=0;
    Terr=0; Tm=[];
    for i=1:nelx
        for j=1:nely
            e=(i-1)*nely+j;
            for i1=ngrid*(i-1)+1:ngrid*i+1
                for j1 = ngrid*(j-1)+1:ngrid*j+1
                    Tm = [Tm;xgnew(j1,i1)];
                    vxPhys(e)=vxPhys(e)+xgnew(j1,i1);
                end
            end
            if min(Tm)>0.001 && max(Tm)<1, Terr=Terr+1; end
            Tm=[];
        end
    end
    vxPhys=vxPhys/(ngrid+1)^2;
    %% CHECK CONVERGENCE
    change=sum(abs(vxnew(:)-vx(:)))/(vol*nely*nelx);
    tol=Terr/nele;
    vx=vxnew;
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%8.4f Vol.:%4.3f ch.:%4.5f Topo.:%7.5f Beta.:%g.\n', loop,-c(loop),mean(vxPhys(:)),change,tol,beta);
    contourf(fnx, flipud(fny), xg-ls, [0 0]);
    axis equal tight off; drawnow
    %% UPDATE HEAVISIDE SMOOTH FUNCTION
    beta=beta+ER;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
